﻿# .SYNOPSIS
#    Creates public folder mailboxes, based on the output of PublicFolderToMailboxMapGenerator.ps1 script, in preparation for Public Folder migration.
#
# .DESCRIPTION
#    The script must be executed on target side of migration. For migration from Exchange 2007 or 2010 to Exchange Online, the script must be executed on a
#    remote PowerShell session created against the Exchange Online organization. For migration from Exchange 2007 or 2010 to Exchange 2013 or 2016 on the
#    same forest, the script must executed on a PowerShell session opened against Exchange 2013 or 2016 servers.
#
#    Copyright (c) 2015 Microsoft Corporation. All rights reserved.
#
#    THIS CODE IS MADE AVAILABLE AS IS, WITHOUT WARRANTY OF ANY KIND. THE ENTIRE RISK
#    OF THE USE OR THE RESULTS FROM THE USE OF THIS CODE REMAINS WITH THE USER.
#
# .PARAMETER FolderMappingCsv
#    Mapping file generated by the PublicFoldertoMailboxMapGenerator.ps1 script.
#
# .PARAMETER EstimatedNumberOfConcurrentUsers
#    Estimated number of simultaneous user connections browsing the Public Folder Hierarchy. This is usually less than the total users in the organization.
#
# .PARAMETER Confirm
#    The Confirm switch causes the script to pause processing and requires you to acknowledge what the script will do before processing continues. You don't have to specify
#    a value with the Confirm switch.
#
# .PARAMETER WhatIf
#    The WhatIf switch instructs the script to simulate the actions that it would take on the object. By using the WhatIf switch, you can view what changes would occur
#    without having to apply any of those changes. You don't have to specify a value with the WhatIf switch.
#
# .EXAMPLE
#    .\Create-PublicFolderMailboxesForMigration.ps1 -FolderMappingCsv mapping.csv -EstimatedNumberOfConcurrentUsers:2000
#
#    This example shows how to invoke the script to create the target public folder mailboxes for migration and also to support 2,000 users browsing the hierarchy at the same time.
#
param(
    [Parameter(Mandatory=$true, HelpMessage = "Mapping file generated by the PublicFoldertoMailboxMapGenerator.ps1 script")]
    [ValidateNotNull()]
    [string] $FolderMappingCsv,

    [Parameter(Mandatory=$true, HelpMessage = "Estimated number of simultaneous user connections browsing the Public Folder Hierarchy")]
    [int] $EstimatedNumberOfConcurrentUsers,

    [Parameter(Mandatory=$false)]
    [bool] $Confirm = $true,

    [Parameter(Mandatory=$false)]
    [switch] $WhatIf = $false
)

function PromptForConfirmation()
{
    param ($title, $message)

    $yes = New-Object System.Management.Automation.Host.ChoiceDescription $LocalizedStrings.ConfirmationYesOption;
    $no = New-Object System.Management.Automation.Host.ChoiceDescription $LocalizedStrings.ConfirmationNoOption;
    [System.Management.Automation.Host.ChoiceDescription[]]$options = $yes, $no;

    $confirmation = $host.ui.PromptForChoice($title, $message, $options, 0);
    if ($confirmation -ne 0)
    {
        Exit;
    }
}

function CreateMailbox()
{
    param($name, $isExcludedFromServingHierarchy, $isMigrationTarget, $isPrimary)

    $mbx = New-Mailbox -PublicFolder $name -IsExcludedFromServingHierarchy:$isExcludedFromServingHierarchy -HoldForMigration:$isPrimary -WhatIf:$WhatIf;
    $outputObj = New-Object psobject;
    $outputObj | add-member noteproperty "Name" ($name);
    $outputObj | add-member noteproperty "IsServingHierarchy" (-not $isExcludedFromServingHierarchy);
    $outputObj | add-member noteproperty "IsMigrationTarget" ($isMigrationTarget);
    $script:mailboxesCreated += $outputObj;

    if (-not $isExcludedFromServingHierarchy)
    {
        $script:totalMailboxesServingHierarchy++;
    }
}

function CreatePrimaryMailbox()
{
    param($name, $isExcludedFromServingHierarchy, $isMigrationTarget)
    CreateMailbox $name $isExcludedFromServingHierarchy $isMigrationTarget $true;
}

function CreateSecondaryMailbox()
{
    param($name, $isExcludedFromServingHierarchy, $isMigrationTarget)
    CreateMailbox $name $isExcludedFromServingHierarchy $isMigrationTarget $false;
}

$MaxConnectionsPerMailbox = 2000;
$MaxMailboxesServingHierarchy = 100;
$script:totalMailboxesServingHierarchy = 0;
$script:mailboxesCreated = @();
$script:mailboxUpdates = @{};

Import-LocalizedData -BindingVariable LocalizedStrings -FileName CreatePublicFolderMailboxesForMigration.strings.psd1

if ($EstimatedNumberOfConcurrentUsers -lt 1)
{
    Write-Error ($LocalizedStrings.InvalidNumberOfConcurrentUsers -f $EstimatedNumberOfConcurrentUsers);
    return;
}

$FolderMappings = Import-Csv $FolderMappingCsv -ErrorAction Stop;
if ($FolderMappings.Count -eq 0)
{
    Write-Error $LocalizedStrings.InvalidCsvEmptyMapping;
    return;
}

# Validate the input CSV and collect the unique mailbox names (multiple folders can map to the same mailbox).
$primaryMailboxName = $null;
$secondaryMailboxesToCreateFromCsv = New-Object 'System.Collections.Generic.HashSet[string]' -ArgumentList ([StringComparer]::CurrentCultureIgnoreCase);
$folderPaths = New-Object 'System.Collections.Generic.HashSet[string]' -ArgumentList ([StringComparer]::CurrentCultureIgnoreCase);
foreach ($folderMapping in $FolderMappings)
{
    if (-not $folderPaths.Add($folderMapping.FolderPath))
    {
        Write-Error ($LocalizedStrings.InvalidCsvDuplicateMapping -f $folderMapping.FolderPath);
        Exit;
    }

    if ($folderMapping.FolderPath -eq "\")
    {
        $primaryMailboxName = $folderMapping.TargetMailbox;
    }
    else
    {
        [void]$secondaryMailboxesToCreateFromCsv.Add($folderMapping.TargetMailbox);
    }
}

if ($primaryMailboxName -eq $null)
{
    Write-Error $LocalizedStrings.InvalidCsvMissingRootFolder;
    Exit;
}

# The order in which primary mailbox appears on the CSV is unpredictable. It may also appear several times mapped
# to different folders. Because of that, it may get added to the secondaryMailboxesToCreateFromCsv collection.
[void]$secondaryMailboxesToCreateFromCsv.Remove($primaryMailboxName);

# Determine the total count of mailboxes to create serving hierarchy
# If primary is the single target for migration, it must serve the hierarchy.
$excludePrimaryFromServingHierarchy = ($secondaryMailboxesToCreateFromCsv.Count -gt 0);
$totalMailboxesServingHierarchyQuota = [Math]::Ceiling($EstimatedNumberOfConcurrentUsers / $MaxConnectionsPerMailbox);
if ($totalMailboxesServingHierarchyQuota -le 1)
{
    # There should be at least one mailbox serving hierarchy.
    $totalMailboxesServingHierarchyQuota = 1;
}
elseif ($totalMailboxesServingHierarchyQuota -gt $MaxMailboxesServingHierarchy)
{
    if ($Confirm)
    {
        PromptForConfirmation $LocalizedStrings.ConfirmTooManyUsersTitle ($LocalizedStrings.ConfirmTooManyUsersMessage -f $MaxMailboxesServingHierarchy);
    }

    $totalMailboxesServingHierarchyQuota = $MaxMailboxesServingHierarchy;
}

$hierarchyMailboxInfo = $(Get-OrganizationConfig).RootPublicFolderMailbox;
if ($hierarchyMailboxInfo.HierarchyMailboxGuid -ne [Guid]::Empty)
{
    if (-not $hierarchyMailboxInfo.LockedForMigration)
    {
        Write-Error $LocalizedStrings.DeploymentNotLockedForMigration;
        Exit;
    }

    $primaryMailbox = Get-Mailbox -PublicFolder $hierarchyMailboxInfo.HierarchyMailboxGuid.ToString() -ErrorAction:Stop;
    if ($primaryMailboxName -ne $primaryMailbox.Name)
    {
        Write-Error ($LocalizedStrings.PrimaryMailboxNameNotMatching -f $primaryMailboxName, $primaryMailbox.Name);
        Exit;
    }

    if ($Confirm)
    {
        PromptForConfirmation $LocalizedStrings.PublicFolderMailboxesAlreadyExistTitle $LocalizedStrings.PublicFolderMailboxesAlreadyExistMessage;
    }

    $secondaryMailboxesServingHierarchyQuota = $totalMailboxesServingHierarchyQuota;
    if ($excludePrimaryFromServingHierarchy)
    {
        # Exclude primary from serving hierarchy when there is at least one other secondary mailbox in the migration plan
        if (-not $primaryMailbox.IsExcludedFromServingHierarchy)
        {
            $script:mailboxUpdates.Add($primaryMailbox.Identity, $true);
        }
    }
    else
    {
        # If primary is the only mailbox in the CSV, it must serve the hierarchy.
        if ($primaryMailbox.IsExcludedFromServingHierarchy)
        {
            $script:mailboxUpdates.Add($primaryMailbox.Identity, $false);
        }

        # Since primary will be serving, reduce the quota for secondary by 1
        $secondaryMailboxesServingHierarchyQuota--;
        $script:totalMailboxesServingHierarchy++;
    }

    $existingMailboxes = @(Get-Mailbox -PublicFolder -ResultSize:Unlimited -ErrorAction:Stop);
    $existingSecondaryMailboxesInfo = @();
    $existingSecondaryInCsvServingHierarchyCount = 0;
    foreach ($existingMailbox in $existingMailboxes)
    {
        if ($primaryMailbox.ExchangeGuid -eq $existingMailbox.ExchangeGuid)
        {
            # Skip primary
            continue;
        }

        $migrationTarget = $secondaryMailboxesToCreateFromCsv.Remove($existingMailbox.Name);
        $mailboxInfo = New-Object psobject -Property @{
            Name = $existingMailbox.Name;
            ServingHierarchy = (-not $existingMailbox.IsExcludedFromServingHierarchy);
            MigrationTarget = $migrationTarget;
        };

        if ($migrationTarget -and `
            $mailboxInfo.ServingHierarchy -and `
            $secondaryMailboxesServingHierarchyQuota -gt 0)
        {
            $secondaryMailboxesServingHierarchyQuota--;
            $script:totalMailboxesServingHierarchy++;
        }
        else
        {
            $existingSecondaryMailboxesInfo += $mailboxInfo;
        }
    }

    if ($secondaryMailboxesServingHierarchyQuota -le $secondaryMailboxesToCreateFromCsv.Count)
    {
        $secondaryMailboxesServingHierarchyQuota = 0;
    }
    else
    {
        $secondaryMailboxesServingHierarchyQuota -= $secondaryMailboxesToCreateFromCsv.Count;
    }

    # Sort the mailboxes in the priority order to serve hierarchy. We should favor mailboxes that are included on the input CSV
    # (i.e. mailboxes that are part of the migration plan). Secondly, we should minimize the number of updates needed to include
    # or exclude existing mailboxes in the hierarchy-serving collection.
    $existingSecondaryMailboxesInfo = @($existingSecondaryMailboxesInfo | sort MigrationTarget,ServingHierarchy -Descending);
    for ($i = 0; $i -lt $existingSecondaryMailboxesInfo.Length; $i++)
    {
        $mailboxInfo = $existingSecondaryMailboxesInfo[$i];
        if ($i -lt $secondaryMailboxesServingHierarchyQuota -and (-not $mailboxInfo.ServingHierarchy))
        {
            # Update to serve hierarchy
            $script:mailboxUpdates.Add($mailboxInfo.Name, $false);
            $script:totalMailboxesServingHierarchy++;
        }
        elseif ($mailboxInfo.ServingHierarchy)
        {
            if ($i -ge $secondaryMailboxesServingHierarchyQuota)
            {
                # Exclude from serving hierarchy
                $script:mailboxUpdates.Add($mailboxInfo.Name, $true);
            }
            else
            {
                $script:totalMailboxesServingHierarchy++;
            }
        }
    }
}

$totalMailboxesToCreate = $secondaryMailboxesToCreateFromCsv.Count;
if ($hierarchyMailboxInfo.HierarchyMailboxGuid -eq [Guid]::Empty)
{
    $totalMailboxesToCreate++;
}

if ($Confirm)
{
    $confirmationMessage = ($LocalizedStrings.ConfirmMailboxOperationsMessage -f ($totalMailboxesToCreate + $additionalCount), $script:mailboxUpdates.Count, $totalMailboxesServingHierarchyQuota);
    PromptForConfirmation $LocalizedStrings.ConfirmMailboxOperationsTitle $confirmationMessage;
}

# Execute operations in the following order:
# 1. Exclude existing mailboxes from serving hierarchy
# 2. Mark existing mailboxes to serve hierarchy (1 and 2 are the reason why we sort the updates by "Value")
# 3. Create the mailboxes that are part of the migration plan
# 4. Create additional mailboxes to serve hierarchy
$updatesExecuted = 0;
foreach ($update in ($script:mailboxUpdates.GetEnumerator() | sort Value -Descending))
{
    Write-Progress -Activity $LocalizedStrings.UpdatingMailboxesActivity `
            -Status ($LocalizedStrings.UpdatingMailboxesProgressStatus -f $updatesExecuted, $script:mailboxUpdates.Count, $update.Key) `
            -PercentComplete (100 * $updatesExecuted / $script:mailboxUpdates.Count);

    Set-Mailbox -PublicFolder -Identity:$update.Key -IsExcludedFromServingHierarchy:$update.Value -ErrorAction:Stop;
    $updatesExecuted++;
}

# Create primary if it doesn't exist yet.
if ($hierarchyMailboxInfo.HierarchyMailboxGuid -eq [Guid]::Empty)
{
    Write-Progress -Activity $LocalizedStrings.CreatingMailboxesForMigrationActivity `
            -Status ($LocalizedStrings.CreatingMailboxesProgressStatus -f 0, $totalMailboxesToCreate, $primaryMailboxName) `
            -PercentComplete 0;

    CreatePrimaryMailbox $primaryMailboxName $excludePrimaryFromServingHierarchy $true;
}
else
{
    Write-Host $LocalizedStrings.SkipPrimaryCreation;
}

$isMigrationTarget = $true;
foreach($secondaryMailboxName in $secondaryMailboxesToCreateFromCsv)
{
    Write-Progress -Activity $LocalizedStrings.CreatingMailboxesForMigrationActivity `
        -Status ($LocalizedStrings.CreatingMailboxesProgressStatus -f $script:mailboxesCreated.Length, $totalMailboxesToCreate, $secondaryMailboxName) `
        -PercentComplete (100 * $script:mailboxesCreated.Length / $totalMailboxesToCreate);

    $excludeFromServingHierarchy = ($script:totalMailboxesServingHierarchy -ge $totalMailboxesServingHierarchyQuota);
    CreateSecondaryMailbox $secondaryMailboxName $excludeFromServingHierarchy $isMigrationTarget;
}

$isMigrationTarget = $false;
$excludeFromServingHierarchy = $false;
$additionalCount = $totalMailboxesServingHierarchyQuota - $script:totalMailboxesServingHierarchy;
for ($i = 0; $i -lt $additionalCount; $i++)
{
    $name = "AutoSplit_" + [Guid]::NewGuid().ToString("n");
    Write-Progress -Activity $LocalizedStrings.CreatingMailboxesToServeHierarchyActivity `
        -Status ($LocalizedStrings.CreatingMailboxesProgressStatus -f $i, $additionalCount, $name) `
        -PercentComplete (100 * $i / $additionalCount);

    CreateSecondaryMailbox $name $excludeFromServingHierarchy $isMigrationTarget;
}

Write-Host ($LocalizedStrings.FinalSummary -f $script:mailboxesCreated.Length, $script:mailboxUpdates.Count, $script:totalMailboxesServingHierarchy);

if ($script:mailboxesCreated.Length -gt 0)
{
    Write-Host $LocalizedStrings.MailboxesCreatedSummary;
    $script:mailboxesCreated | ft -a;
}
# SIG # Begin signature block
# MIId3AYJKoZIhvcNAQcCoIIdzTCCHckCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUbw5cw8VzzwoPeFHhnALpjtBt
# YM+gghhlMIIEwzCCA6ugAwIBAgITMwAAAMZ4gDYBdRppcgAAAAAAxjANBgkqhkiG
# 9w0BAQUFADB3MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
# A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
# HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EwHhcNMTYwOTA3MTc1ODUz
# WhcNMTgwOTA3MTc1ODUzWjCBszELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
# bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
# b3JhdGlvbjENMAsGA1UECxMETU9QUjEnMCUGA1UECxMebkNpcGhlciBEU0UgRVNO
# OkY1MjgtMzc3Ny04QTc2MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBT
# ZXJ2aWNlMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEArQsjG6jKiCgU
# NuPDaF0GhCh1QYcSqJypNAJgoa1GtgoNrKXTDUZF6K+eHPNzXv9v/LaYLZX2GyOI
# 9lGz55tXVv1Ny6I1ueVhy2cUAhdE+IkVR6AtCo8Ar8uHwEpkyTi+4Ywr6sOGM7Yr
# wBqw+SeaBjBwON+8E8SAz0pgmHHj4cNvt5A6R+IQC6tyiFx+JEMO1qqnITSI2qx3
# kOXhD3yTF4YjjRnTx3HGpfawUCyfWsxasAHHlILEAfsVAmXsbr4XAC2HBZGKXo03
# jAmfvmbgbm3V4KBK296Unnp92RZmwAEqL08n+lrl+PEd6w4E9mtFHhR9wGSW29C5
# /0bOar9zHwIDAQABo4IBCTCCAQUwHQYDVR0OBBYEFNS/9jKwiDEP5hmU8T6/Mfpb
# Ag8JMB8GA1UdIwQYMBaAFCM0+NlSRnAK7UD7dvuzK7DDNbMPMFQGA1UdHwRNMEsw
# SaBHoEWGQ2h0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3Rz
# L01pY3Jvc29mdFRpbWVTdGFtcFBDQS5jcmwwWAYIKwYBBQUHAQEETDBKMEgGCCsG
# AQUFBzAChjxodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jv
# c29mdFRpbWVTdGFtcFBDQS5jcnQwEwYDVR0lBAwwCgYIKwYBBQUHAwgwDQYJKoZI
# hvcNAQEFBQADggEBAJhbANzvo0iL5FA5Z5QkwG+PvkDfOaYsTYksqFk+MgpqzPxc
# FwSYME/S/wyihd4lwgQ6CPdO5AGz3m5DZU7gPS5FcCl10k9pTxZ4s857Pu8ZrE2x
# rnUyUiQFl5DYSNroRPuQYRZZXs2xK1WVn1JcwcAwJwfu1kwnebPD90o1DRlNozHF
# 3NMaIo0nCTRAN86eSByKdYpDndgpVLSoN2wUnsh4bLcZqod4ozdkvgGS7N1Af18R
# EFSUBVraf7MoSxKeNIKLLyhgNxDxZxrUgnPb3zL73zOj40A1Ibw3WzJob8vYK+gB
# YWORl4jm6vCwAq/591z834HDNH60Ud0bH+xS7PowggYHMIID76ADAgECAgphFmg0
# AAAAAAAcMA0GCSqGSIb3DQEBBQUAMF8xEzARBgoJkiaJk/IsZAEZFgNjb20xGTAX
# BgoJkiaJk/IsZAEZFgltaWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
# IENlcnRpZmljYXRlIEF1dGhvcml0eTAeFw0wNzA0MDMxMjUzMDlaFw0yMTA0MDMx
# MzAzMDlaMHcxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xITAf
# BgNVBAMTGE1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQTCCASIwDQYJKoZIhvcNAQEB
# BQADggEPADCCAQoCggEBAJ+hbLHf20iSKnxrLhnhveLjxZlRI1Ctzt0YTiQP7tGn
# 0UytdDAgEesH1VSVFUmUG0KSrphcMCbaAGvoe73siQcP9w4EmPCJzB/LMySHnfL0
# Zxws/HvniB3q506jocEjU8qN+kXPCdBer9CwQgSi+aZsk2fXKNxGU7CG0OUoRi4n
# rIZPVVIM5AMs+2qQkDBuh/NZMJ36ftaXs+ghl3740hPzCLdTbVK0RZCfSABKR2YR
# JylmqJfk0waBSqL5hKcRRxQJgp+E7VV4/gGaHVAIhQAQMEbtt94jRrvELVSfrx54
# QTF3zJvfO4OToWECtR0Nsfz3m7IBziJLVP/5BcPCIAsCAwEAAaOCAaswggGnMA8G
# A1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFCM0+NlSRnAK7UD7dvuzK7DDNbMPMAsG
# A1UdDwQEAwIBhjAQBgkrBgEEAYI3FQEEAwIBADCBmAYDVR0jBIGQMIGNgBQOrIJg
# QFYnl+UlE/wq4QpTlVnkpKFjpGEwXzETMBEGCgmSJomT8ixkARkWA2NvbTEZMBcG
# CgmSJomT8ixkARkWCW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJvb3Qg
# Q2VydGlmaWNhdGUgQXV0aG9yaXR5ghB5rRahSqClrUxzWPQHEy5lMFAGA1UdHwRJ
# MEcwRaBDoEGGP2h0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1
# Y3RzL21pY3Jvc29mdHJvb3RjZXJ0LmNybDBUBggrBgEFBQcBAQRIMEYwRAYIKwYB
# BQUHMAKGOGh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljcm9z
# b2Z0Um9vdENlcnQuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0GCSqGSIb3DQEB
# BQUAA4ICAQAQl4rDXANENt3ptK132855UU0BsS50cVttDBOrzr57j7gu1BKijG1i
# uFcCy04gE1CZ3XpA4le7r1iaHOEdAYasu3jyi9DsOwHu4r6PCgXIjUji8FMV3U+r
# kuTnjWrVgMHmlPIGL4UD6ZEqJCJw+/b85HiZLg33B+JwvBhOnY5rCnKVuKE5nGct
# xVEO6mJcPxaYiyA/4gcaMvnMMUp2MT0rcgvI6nA9/4UKE9/CCmGO8Ne4F+tOi3/F
# NSteo7/rvH0LQnvUU3Ih7jDKu3hlXFsBFwoUDtLaFJj1PLlmWLMtL+f5hYbMUVbo
# nXCUbKw5TNT2eb+qGHpiKe+imyk0BncaYsk9Hm0fgvALxyy7z0Oz5fnsfbXjpKh0
# NbhOxXEjEiZ2CzxSjHFaRkMUvLOzsE1nyJ9C/4B5IYCeFTBm6EISXhrIniIh0EPp
# K+m79EjMLNTYMoBMJipIJF9a6lbvpt6Znco6b72BJ3QGEe52Ib+bgsEnVLaxaj2J
# oXZhtG6hE6a/qkfwEm/9ijJssv7fUciMI8lmvZ0dhxJkAj0tr1mPuOQh5bWwymO0
# eFQF1EEuUKyUsKV4q7OglnUa2ZKHE3UiLzKoCG6gW4wlv6DvhMoh1useT8ma7kng
# 9wFlb4kLfchpyOZu6qeXzjEp/w7FW1zYTRuh2Povnj8uVRZryROj/TCCBhEwggP5
# oAMCAQICEzMAAACOh5GkVxpfyj4AAAAAAI4wDQYJKoZIhvcNAQELBQAwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMTAeFw0xNjExMTcyMjA5MjFaFw0xODAy
# MTcyMjA5MjFaMIGDMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
# MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
# MQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNyb3NvZnQgQ29ycG9yYXRpb24w
# ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDQh9RCK36d2cZ61KLD4xWS
# 0lOdlRfJUjb6VL+rEK/pyefMJlPDwnO/bdYA5QDc6WpnNDD2Fhe0AaWVfIu5pCzm
# izt59iMMeY/zUt9AARzCxgOd61nPc+nYcTmb8M4lWS3SyVsK737WMg5ddBIE7J4E
# U6ZrAmf4TVmLd+ArIeDvwKRFEs8DewPGOcPUItxVXHdC/5yy5VVnaLotdmp/ZlNH
# 1UcKzDjejXuXGX2C0Cb4pY7lofBeZBDk+esnxvLgCNAN8mfA2PIv+4naFfmuDz4A
# lwfRCz5w1HercnhBmAe4F8yisV/svfNQZ6PXlPDSi1WPU6aVk+ayZs/JN2jkY8fP
# AgMBAAGjggGAMIIBfDAfBgNVHSUEGDAWBgorBgEEAYI3TAgBBggrBgEFBQcDAzAd
# BgNVHQ4EFgQUq8jW7bIV0qqO8cztbDj3RUrQirswUgYDVR0RBEswSaRHMEUxDTAL
# BgNVBAsTBE1PUFIxNDAyBgNVBAUTKzIzMDAxMitiMDUwYzZlNy03NjQxLTQ0MWYt
# YmM0YS00MzQ4MWU0MTVkMDgwHwYDVR0jBBgwFoAUSG5k5VAF04KqFzc3IrVtqMp1
# ApUwVAYDVR0fBE0wSzBJoEegRYZDaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3Br
# aW9wcy9jcmwvTWljQ29kU2lnUENBMjAxMV8yMDExLTA3LTA4LmNybDBhBggrBgEF
# BQcBAQRVMFMwUQYIKwYBBQUHMAKGRWh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9w
# a2lvcHMvY2VydHMvTWljQ29kU2lnUENBMjAxMV8yMDExLTA3LTA4LmNydDAMBgNV
# HRMBAf8EAjAAMA0GCSqGSIb3DQEBCwUAA4ICAQBEiQKsaVPzxLa71IxgU+fKbKhJ
# aWa+pZpBmTrYndJXAlFq+r+bltumJn0JVujc7SV1eqVHUqgeSxZT8+4PmsMElSnB
# goSkVjH8oIqRlbW/Ws6pAR9kRqHmyvHXdHu/kghRXnwzAl5RO5vl2C5fAkwJnBpD
# 2nHt5Nnnotp0LBet5Qy1GPVUCdS+HHPNIHuk+sjb2Ns6rvqQxaO9lWWuRi1XKVjW
# kvBs2mPxjzOifjh2Xt3zNe2smjtigdBOGXxIfLALjzjMLbzVOWWplcED4pLJuavS
# Vwqq3FILLlYno+KYl1eOvKlZbiSSjoLiCXOC2TWDzJ9/0QSOiLjimoNYsNSa5jH6
# lEeOfabiTnnz2NNqMxZQcPFCu5gJ6f/MlVVbCL+SUqgIxPHo8f9A1/maNp39upCF
# 0lU+UK1GH+8lDLieOkgEY+94mKJdAw0C2Nwgq+ZWtd7vFmbD11WCHk+CeMmeVBoQ
# YLcXq0ATka6wGcGaM53uMnLNZcxPRpgtD1FgHnz7/tvoB3kH96EzOP4JmtuPe7Y6
# vYWGuMy8fQEwt3sdqV0bvcxNF/duRzPVQN9qyi5RuLW5z8ME0zvl4+kQjOunut6k
# LjNqKS8USuoewSI4NQWF78IEAA1rwdiWFEgVr35SsLhgxFK1SoK3hSoASSomgyda
# Qd691WZJvAuceHAJvDCCB3owggVioAMCAQICCmEOkNIAAAAAAAMwDQYJKoZIhvcN
# AQELBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAw
# BgNVBAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eSAyMDEx
# MB4XDTExMDcwODIwNTkwOVoXDTI2MDcwODIxMDkwOVowfjELMAkGA1UEBhMCVVMx
# EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoT
# FU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9zb2Z0IENvZGUg
# U2lnbmluZyBQQ0EgMjAxMTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIB
# AKvw+nIQHC6t2G6qghBNNLrytlghn0IbKmvpWlCquAY4GgRJun/DDB7dN2vGEtgL
# 8DjCmQawyDnVARQxQtOJDXlkh36UYCRsr55JnOloXtLfm1OyCizDr9mpK656Ca/X
# llnKYBoF6WZ26DJSJhIv56sIUM+zRLdd2MQuA3WraPPLbfM6XKEW9Ea64DhkrG5k
# NXimoGMPLdNAk/jj3gcN1Vx5pUkp5w2+oBN3vpQ97/vjK1oQH01WKKJ6cuASOrdJ
# Xtjt7UORg9l7snuGG9k+sYxd6IlPhBryoS9Z5JA7La4zWMW3Pv4y07MDPbGyr5I4
# ftKdgCz1TlaRITUlwzluZH9TupwPrRkjhMv0ugOGjfdf8NBSv4yUh7zAIXQlXxgo
# tswnKDglmDlKNs98sZKuHCOnqWbsYR9q4ShJnV+I4iVd0yFLPlLEtVc/JAPw0Xpb
# L9Uj43BdD1FGd7P4AOG8rAKCX9vAFbO9G9RVS+c5oQ/pI0m8GLhEfEXkwcNyeuBy
# 5yTfv0aZxe/CHFfbg43sTUkwp6uO3+xbn6/83bBm4sGXgXvt1u1L50kppxMopqd9
# Z4DmimJ4X7IvhNdXnFy/dygo8e1twyiPLI9AN0/B4YVEicQJTMXUpUMvdJX3bvh4
# IFgsE11glZo+TzOE2rCIF96eTvSWsLxGoGyY0uDWiIwLAgMBAAGjggHtMIIB6TAQ
# BgkrBgEEAYI3FQEEAwIBADAdBgNVHQ4EFgQUSG5k5VAF04KqFzc3IrVtqMp1ApUw
# GQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB
# /wQFMAMBAf8wHwYDVR0jBBgwFoAUci06AjGQQ7kUBU7h6qfHMdEjiTQwWgYDVR0f
# BFMwUTBPoE2gS4ZJaHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJv
# ZHVjdHMvTWljUm9vQ2VyQXV0MjAxMV8yMDExXzAzXzIyLmNybDBeBggrBgEFBQcB
# AQRSMFAwTgYIKwYBBQUHMAKGQmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kv
# Y2VydHMvTWljUm9vQ2VyQXV0MjAxMV8yMDExXzAzXzIyLmNydDCBnwYDVR0gBIGX
# MIGUMIGRBgkrBgEEAYI3LgMwgYMwPwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cubWlj
# cm9zb2Z0LmNvbS9wa2lvcHMvZG9jcy9wcmltYXJ5Y3BzLmh0bTBABggrBgEFBQcC
# AjA0HjIgHQBMAGUAZwBhAGwAXwBwAG8AbABpAGMAeQBfAHMAdABhAHQAZQBtAGUA
# bgB0AC4gHTANBgkqhkiG9w0BAQsFAAOCAgEAZ/KGpZjgVHkaLtPYdGcimwuWEeFj
# kplCln3SeQyQwWVfLiw++MNy0W2D/r4/6ArKO79HqaPzadtjvyI1pZddZYSQfYtG
# UFXYDJJ80hpLHPM8QotS0LD9a+M+By4pm+Y9G6XUtR13lDni6WTJRD14eiPzE32m
# kHSDjfTLJgJGKsKKELukqQUMm+1o+mgulaAqPyprWEljHwlpblqYluSD9MCP80Yr
# 3vw70L01724lruWvJ+3Q3fMOr5kol5hNDj0L8giJ1h/DMhji8MUtzluetEk5CsYK
# wsatruWy2dsViFFFWDgycScaf7H0J/jeLDogaZiyWYlobm+nt3TDQAUGpgEqKD6C
# PxNNZgvAs0314Y9/HG8VfUWnduVAKmWjw11SYobDHWM2l4bf2vP48hahmifhzaWX
# 0O5dY0HjWwechz4GdwbRBrF1HxS+YWG18NzGGwS+30HHDiju3mUv7Jf2oVyW2ADW
# oUa9WfOXpQlLSBCZgB/QACnFsZulP0V3HjXG0qKin3p6IvpIlR+r+0cjgPWe+L9r
# t0uX4ut1eBrs6jeZeRhL/9azI2h15q/6/IvrC4DqaTuv/DDtBEyO3991bWORPdGd
# Vk5Pv4BXIqF4ETIheu9BCrE/+6jMpF3BoYibV3FWTkhFwELJm3ZbCoBIa/15n8G9
# bW1qyVJzEw16UM0xggThMIIE3QIBATCBlTB+MQswCQYDVQQGEwJVUzETMBEGA1UE
# CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
# b2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBTaWduaW5n
# IFBDQSAyMDExAhMzAAAAjoeRpFcaX8o+AAAAAACOMAkGBSsOAwIaBQCggfUwGQYJ
# KoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisGAQQB
# gjcCARUwIwYJKoZIhvcNAQkEMRYEFBO8Wc/Gk0Swp/e7MgDipTiwKmVyMIGUBgor
# BgEEAYI3AgEMMYGFMIGCoFqAWABDAHIAZQBhAHQAZQAtAFAAdQBiAGwAaQBjAEYA
# bwBsAGQAZQByAE0AYQBpAGwAYgBvAHgAZQBzAEYAbwByAE0AaQBnAHIAYQB0AGkA
# bwBuAC4AcABzADGhJIAiaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL2V4Y2hhbmdl
# IDANBgkqhkiG9w0BAQEFAASCAQDFvSSl5NLFvvI8ncEkwQr77dGx60AdVZMvV8iM
# QAXlbWE58t42cRGlNTA9wr7VG/mqfIw1MprIK1u1iWA5bIubUMeyDYRKBOShiVfw
# B9vc1KpIGOu91pLUl9WApmwmKjI8o2bov0uDKS4Fh46FzHMFYOJHqVdNctQWwODv
# +vIhFB558nNwXzVN6wlHDlDg9nI3eysUsgMNnQt6n0QNLe/59vphbiJoT9vzhUbc
# 2tWpY/a4Aimn8yaWITl+HYQI6B1vt1QbfE6dibAK1kNdxqjbKspn2khzVKN14bcg
# ezMOJdOaWup9P+cxMZAIpDAFOjLgPRz0sgfMjBFkTgkONH+GoYICKDCCAiQGCSqG
# SIb3DQEJBjGCAhUwggIRAgEBMIGOMHcxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
# YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
# Q29ycG9yYXRpb24xITAfBgNVBAMTGE1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQQIT
# MwAAAMZ4gDYBdRppcgAAAAAAxjAJBgUrDgMCGgUAoF0wGAYJKoZIhvcNAQkDMQsG
# CSqGSIb3DQEHATAcBgkqhkiG9w0BCQUxDxcNMTcwMzMxMTQ0OTMyWjAjBgkqhkiG
# 9w0BCQQxFgQUCW5XtWmkZJh2nMlTAH6sjZe/2tkwDQYJKoZIhvcNAQEFBQAEggEA
# c0/dqv9NomuaYvDSuqlt8naoxErDUYT52MC+0VrBZMeRuPqQbx4SFO778qsG/lZN
# A4MN5g/10qRY0JmvRnygFWS+IdSWmqdHxvM6wxN3kRZn2xHjLWSB2zHv57lJe59K
# JJ1FMZU5oupfWWmTGXpj0YTyYS9UZRqlTcpuFVJeyNc/ol/OGP413o0aMJ7TBVXs
# qz2d+iuG07ekEDRJNtV5+jDYwg/yghxJRXgQQnzqvrE2ShBrH0syuTe2ZuaJun3o
# 9RZISDJ5ycfelF8pRepMxme+Cyo/WGGotZR34ayrsup+ymHqnLLP/pzZ/b5AAuRP
# pk5SkmAETgalc3IQPru+CQ==
# SIG # End signature block
